const fileInput = document.getElementById('fileInput');
const previewImg = document.getElementById('previewImg');
const consent = document.getElementById('consent');
const analyzeBtn = document.getElementById('analyzeBtn');
const results = document.getElementById('results');
const profileEl = document.getElementById('profile');
const swatchesEl = document.getElementById('swatches');
const keywordsEl = document.getElementById('keywords');
const canvas = document.getElementById('canvas');
const sephoraCards = document.getElementById('sephoraCards');
const ultaCards = document.getElementById('ultaCards');
const sephoraLink = document.getElementById('sephoraLink');
const ultaLink = document.getElementById('ultaLink');

let imageDataUrl = null;

fileInput.addEventListener('change', (e) => {
  const file = e.target.files[0];
  if (!file) return;
  const reader = new FileReader();
  reader.onload = () => {
    imageDataUrl = reader.result;
    previewImg.src = imageDataUrl;
    previewImg.style.display = 'block';
    maybeEnable();
  };
  reader.readAsDataURL(file);
});

consent.addEventListener('change', maybeEnable);

function maybeEnable() {
  analyzeBtn.disabled = !(imageDataUrl && consent.checked);
}

analyzeBtn.addEventListener('click', async () => {
  if (!imageDataUrl) return;
  const colors = await extractColors(imageDataUrl);
  const profile = buildProfile(colors.average);
  renderProfile(profile, colors.samples);
  const query = `${profile.season} ${profile.tone} ${profile.colorName} lipstick`;
  keywordsEl.textContent = `Search keywords: ${query}`;
  await Promise.all([
    loadSephora(query),
    loadUlta(query)
  ]);
});

function extractColors(dataUrl) {
  return new Promise((resolve) => {
    const img = new Image();
    img.onload = () => {
      const size = 160;
      canvas.width = size;
      canvas.height = size;
      const ctx = canvas.getContext('2d');
      ctx.drawImage(img, 0, 0, size, size);
      const { data } = ctx.getImageData(0, 0, size, size);
      const samples = [];
      let r = 0, g = 0, b = 0, count = 0;
      for (let i = 0; i < data.length; i += 40) { // skip some pixels
        const pr = data[i];
        const pg = data[i + 1];
        const pb = data[i + 2];
        const alpha = data[i + 3];
        if (alpha < 200) continue;
        r += pr; g += pg; b += pb; count++;
        if (samples.length < 5) samples.push({ r: pr, g: pg, b: pb });
      }
      const avg = { r: Math.round(r / count), g: Math.round(g / count), b: Math.round(b / count) };
      resolve({ average: avg, samples });
    };
    img.src = dataUrl;
  });
}

function buildProfile(avg) {
  const { h, l } = rgbToHsl(avg.r, avg.g, avg.b);
  const warm = (h < 60 || h > 300);
  const tone = l > 55 ? 'light' : 'deep';
  const season = warm ? (tone === 'light' ? 'spring' : 'autumn') : (tone === 'light' ? 'summer' : 'winter');
  const colorName = hueToName(h);
  return { warm, tone, season, colorName };
}

function hueToName(h) {
  if (h < 15 || h >= 345) return 'red';
  if (h < 45) return 'coral';
  if (h < 70) return 'peach';
  if (h < 150) return 'green';
  if (h < 200) return 'teal';
  if (h < 250) return 'blue';
  if (h < 290) return 'purple';
  return 'rose';
}

function rgbToHsl(r, g, b) {
  r /= 255; g /= 255; b /= 255;
  const max = Math.max(r, g, b), min = Math.min(r, g, b);
  let h, s, l = (max + min) / 2;
  if (max === min) { h = s = 0; }
  else {
    const d = max - min;
    s = l > 0.5 ? d / (2 - max - min) : d / (max + min);
    switch (max) {
      case r: h = (g - b) / d + (g < b ? 6 : 0); break;
      case g: h = (b - r) / d + 2; break;
      default: h = (r - g) / d + 4; break;
    }
    h *= 60;
  }
  return { h, s, l: l * 100 };
}

function renderProfile(profile, samples) {
  results.hidden = false;
  const temp = profile.warm ? 'Warm' : 'Cool';
  profileEl.textContent = `${temp} • ${profile.tone} • ${profile.season}`;
  swatchesEl.innerHTML = '';
  samples.forEach((c) => {
    const sw = document.createElement('div');
    sw.className = 'swatch';
    sw.style.background = `rgb(${c.r}, ${c.g}, ${c.b})`;
    swatchesEl.appendChild(sw);
  });
}

async function loadSephora(query) {
  const url = `https://www.sephora.com/search?keyword=${encodeURIComponent(query)}`;
  sephoraLink.href = url;
  sephoraCards.innerHTML = '<div class="desc">Loading...</div>';
  try {
    const res = await fetch(url);
    const html = await res.text();
    const products = parseSephora(html);
    renderCards(sephoraCards, products, 'Sephora');
  } catch (e) {
    sephoraCards.innerHTML = '<div class="desc">Unable to load. Open search instead.</div>';
  }
}

async function loadUlta(query) {
  const url = `https://www.ulta.com/search?searchTerm=${encodeURIComponent(query)}`;
  ultaLink.href = url;
  ultaCards.innerHTML = '<div class="desc">Loading...</div>';
  try {
    const res = await fetch(url);
    const html = await res.text();
    const products = parseUlta(html);
    renderCards(ultaCards, products, 'Ulta');
  } catch (e) {
    ultaCards.innerHTML = '<div class="desc">Unable to load. Open search instead.</div>';
  }
}

function renderCards(container, products, vendor) {
  container.innerHTML = '';
  if (!products.length) {
    container.innerHTML = `<div class="desc">No products parsed. Try opening ${vendor} search.</div>`;
    return;
  }
  products.slice(0, 6).forEach((p) => {
    const card = document.createElement('a');
    card.className = 'card';
    card.href = p.link;
    card.target = '_blank';
    card.rel = 'noreferrer';
    card.innerHTML = `
      <img src="${p.image}" alt="${p.name}" />
      <div class="name">${p.name}</div>
      <div class="rating">${p.rating || ''}</div>
      <div class="desc">${p.desc || ''}</div>
    `;
    container.appendChild(card);
  });
}

function parseSephora(html) {
  const doc = new DOMParser().parseFromString(html, 'text/html');
  const items = [...doc.querySelectorAll('[data-at="product_item"]')];
  return items.map((item) => {
    const name = item.querySelector('[data-at="product_name"]')?.textContent?.trim();
    const desc = item.querySelector('[data-at="product_description"]')?.textContent?.trim();
    const rating = item.querySelector('[data-at="rating"]')?.textContent?.trim();
    const link = item.querySelector('a')?.href;
    const image = item.querySelector('img')?.src;
    return name && link ? { name, desc, rating, link, image } : null;
  }).filter(Boolean);
}

function parseUlta(html) {
  const doc = new DOMParser().parseFromString(html, 'text/html');
  const items = [...doc.querySelectorAll('[data-testid="product-card"], .ProductListing .ProductListingGrid')];
  return items.map((item) => {
    const name = item.querySelector('[data-testid="product-card-title"], .ProductListingTitle')?.textContent?.trim();
    const desc = item.querySelector('[data-testid="product-card-brand"], .ProductListingBrand')?.textContent?.trim();
    const rating = item.querySelector('[data-testid="product-card-rating"], .ReviewSummary')?.textContent?.trim();
    const link = item.querySelector('a')?.href;
    const image = item.querySelector('img')?.src;
    return name && link ? { name, desc, rating, link, image } : null;
  }).filter(Boolean);
}

const tabs = document.querySelectorAll('.tab');
const tabContents = document.querySelectorAll('.tab-content');

tabs.forEach((tab) => {
  tab.addEventListener('click', () => {
    tabs.forEach((t) => t.classList.remove('active'));
    tabContents.forEach((c) => c.classList.remove('active'));
    tab.classList.add('active');
    document.getElementById(tab.dataset.tab).classList.add('active');
  });
});
